import os, subprocess, shlex, tempfile, textwrap, csv

AFTER_EFFECTS_APP = "Adobe After Effects 2025"

# JSX to run
JSX_BATCH = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Rotate3Dtext_stroke_fully_compatible.jsx"

PROJECT   = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Rotate3Dtext.aep"
CSV_PATH  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/teams_rgb_correct.csv"

# AE/Template settings
COMP        = "Comp 3"
LAYER       = "FIELD GOAL"
ANIM_NAME   = "FG_2_6"
COLORSET    = "primary"
STROKE_W    = "2"
STROKE_OVER = "1"

# Output
OUTDIR        = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/FG_Simple_384"
PATH_TEMPLATE = "{league}"
OM_TEMPLATE   = "H.264 - Match Source - High Bitrate"
EXT           = ".mp4"

# Modes
RUN_MODE = "league"     # "league" or "subset"
LEAGUE   = "NFL"
SUBSET   = ["BUF", "DAL", "PHI"]

def teams_from_csv(csv_path, league):
    with open(csv_path, newline="") as f:
        sample = f.read(4096); f.seek(0)
        dialect = csv.Sniffer().sniff(sample, delimiters=",\t")
        r = csv.DictReader(f, dialect=dialect)
        out = []
        for row in r:
            lg = (row.get("league") or row.get("League") or row.get("LEAGUE") or "").strip().lower()
            if lg == league.lower():
                abbr = (row.get("abbreviation") or row.get("abbr") or row.get("team") or "").strip().upper()
                if abbr:
                    out.append(abbr)
    return out

def _make_env(league=None, teams=None):
    env = {
        "AE_PROJECT": PROJECT,
        "AE_CSV": CSV_PATH,
        "AE_COMP": COMP,
        "AE_LAYER": LAYER,
        "AE_ANIM": ANIM_NAME,
        "AE_COLORSET": COLORSET,
        "AE_STROKE_WIDTH": STROKE_W,
        "AE_STROKE_OVER": STROKE_OVER,
        "AE_OUTDIR": OUTDIR,
        "AE_PATH_TEMPLATE": PATH_TEMPLATE,
        "AE_OM_TEMPLATE": OM_TEMPLATE,
        "AE_EXT": EXT,
        "AE_RS_TEMPLATE": "Best Settings",
        "AE_QUIT": "1",
    }
    if league:
        env["AE_LEAGUE"] = league
    if teams:
        env["AE_TEAMS"] = ",".join(teams)
    return env

def _write_bridge_jsx(env: dict) -> str:
    def esc(s: str) -> str:
        return s.replace("\\", "\\\\").replace('"', '\\"')
    lines = [f'$.setenv("{esc(k)}","{esc(str(v))}");' for k, v in env.items()]
    lines.append(f'$.evalFile(File("{esc(JSX_BATCH)}"));')
    code = "\n".join(lines) + "\n"
    tmp = tempfile.NamedTemporaryFile(prefix="ae_bridge_", suffix=".jsx", delete=False)
    tmp.write(code.encode("utf-8")); tmp.flush(); tmp.close()
    print("---- Bridge JSX content ----\n" + code + "-----------------------------")
    return tmp.name

def run_afterfx(env: dict):
    os.makedirs(OUTDIR, exist_ok=True)
    bridge = _write_bridge_jsx(env)
    osa = [
        "osascript",
        "-e", f'tell application "{AFTER_EFFECTS_APP}" to activate',
        "-e", f'tell application "{AFTER_EFFECTS_APP}" to DoScriptFile POSIX file "{bridge}"'
    ]
    print("Running:", " ".join(shlex.quote(c) for c in osa))
    subprocess.run(osa, check=True)

def run_for_league(league):
    nfl_teams = teams_from_csv(CSV_PATH, league)
    if not nfl_teams:
        raise RuntimeError(f"No teams found in CSV for league={league}")
    env = _make_env(league=league, teams=nfl_teams)  # pass both AE_LEAGUE + AE_TEAMS
    run_afterfx(env)

def run_for_subset(teams):
    env = _make_env(teams=teams)
    run_afterfx(env)

if __name__ == "__main__":
    print(f"Using JSX file: {JSX_BATCH}")
    assert os.path.isfile(JSX_BATCH), f"JSX not found: {JSX_BATCH}"
    assert os.path.isfile(PROJECT),   f"Project not found: {PROJECT}"
    assert os.path.isfile(CSV_PATH),  f"CSV not found: {CSV_PATH}"

    if RUN_MODE == "league":
        run_for_league(LEAGUE)
    else:
        run_for_subset(SUBSET)
